package com.ejianc.business.supbusiness.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.ejianc.business.supbusiness.bean.DeliveryEntity;
import com.ejianc.business.supbusiness.bean.OrderDetailEntity;
import com.ejianc.business.supbusiness.bean.OrderEntity;
import com.ejianc.business.supbusiness.constant.InteractiveConstants;
import com.ejianc.business.supbusiness.enums.CloseFlagEnum;
import com.ejianc.business.supbusiness.enums.DeliverCheckStateEnum;
import com.ejianc.business.supbusiness.enums.OrderDeliverStateEnum;
import com.ejianc.business.supbusiness.enums.OrderReceiveStateEnum;
import com.ejianc.business.supbusiness.mapper.OrderMapper;
import com.ejianc.business.supbusiness.service.IDeliveryService;
import com.ejianc.business.supbusiness.service.IOrderDetailService;
import com.ejianc.business.supbusiness.service.IOrderService;
import com.ejianc.business.supbusiness.utils.CommonUtils;
import com.ejianc.business.supbusiness.utils.ProMaterialFileUtils;
import com.ejianc.business.supbusiness.vo.DeliveryVO;
import com.ejianc.business.supbusiness.vo.OrderDetailVO;
import com.ejianc.business.supbusiness.vo.OrderVO;
import com.ejianc.foundation.file.api.IAttachmentApi;
import com.ejianc.foundation.file.vo.AttachmentVO;
import com.ejianc.foundation.share.utils.FileUtil;
import com.ejianc.framework.core.context.InvocationInfoProxy;
import com.ejianc.framework.core.exception.BusinessException;
import com.ejianc.framework.core.kit.mapper.BeanMapper;
import com.ejianc.framework.core.response.BillStateEnum;
import com.ejianc.framework.core.response.CommonResponse;
import com.ejianc.framework.core.response.Parameter;
import com.ejianc.framework.core.response.QueryParam;
import com.ejianc.framework.skeleton.dataPush.ISystemDataPushService;
import com.ejianc.framework.skeleton.template.BaseServiceImpl;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.multipart.MultipartFile;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 消耗材订单表
 *
 * @author generator
 */
@Service("orderService")
public class OrderServiceImpl extends BaseServiceImpl<OrderMapper, OrderEntity> implements IOrderService {
    private Logger logger = LoggerFactory.getLogger(getClass());

    @Autowired
    private ISystemDataPushService systemDataPushService;
    @Autowired
    private IDeliveryService deliveryService;
    @Autowired
    private IOrderDetailService orderDetailService;
    @Autowired
    private IAttachmentApi attachmentApi;

//    @Value("${proMaterial.pushMaterialOrder}")
//    private String pushMaterialOrder;

    /**
     * 单据类型
     */
    private static final String materialOrderBillType = "EJCBT202408000003";
    /**
     * 上传文件单据类型
     */
    private static final String fileSourceType = "materialOrder";

    /**
     * 保存施工方推送的订单数据
     *
     * @param request 施工方请求
     * @param file    文件信息
     * @return 保存结果
     */
    @Override
    public boolean saveOrder(HttpServletRequest request, MultipartFile file) {
        logger.info("进入订单保存接口>>>>>>>>>>>>>>>>>>>>>>>>");
        String order = request.getParameter("order");
        logger.info("接收到数据：{}", order);
        OrderVO saveOrUpdateVO = JSON.parseObject(order, new TypeReference<OrderVO>() {
        });
        OrderEntity entity = BeanMapper.map(saveOrUpdateVO, OrderEntity.class);
        if (CollectionUtils.isEmpty(entity.getOrderDetailList())) {
            throw new BusinessException("材料列表不能为空！");
        }
//        // 查询是否存在相同sourceId的订单
//        Long sourceId = entity.getSourceId();
//        QueryParam queryParam = new QueryParam();
//        queryParam.getParams().put("sourceId", new Parameter(QueryParam.EQ, sourceId));
//        List<OrderEntity> orderEntityList = super.queryList(queryParam);
//        // 同一个订单重复推送，则是抛出异常
//        if (CollectionUtils.isNotEmpty(orderEntityList)) {
//            logger.info("存在相同sourceID的数据，原数据:{}", JSONObject.toJSONString(orderEntityList));
//            throw new BusinessException("该订单已在存在！");
//        }
        OrderEntity checkEntity = super.selectById(saveOrUpdateVO.getId());
        // 同一个订单重复推送，则是抛出异常
        if (null != checkEntity) {
            logger.info("存在相同sourceID的数据，原数据:{}", JSONObject.toJSONString(checkEntity));
//            throw new BusinessException("订单已在存在！");
        }
        entity.setOrderFlag(CloseFlagEnum.NORMAL.getCode());
        entity.setBillState(BillStateEnum.UNCOMMITED_STATE.getBillStateCode());
        entity.setReceiveState(OrderReceiveStateEnum.WAIT_RECEIVE.getCode());
        // 处理附件
        //保存单据中附件并获取到上传后附件的Id
        if (null != file) {
            // 供方id与施工方id相同，此时传入数据中存在id
            List<Long> attrs = ProMaterialFileUtils
                    .upFile(fileSourceType, entity.getId().toString(), materialOrderBillType, file, request, logger,
                            FileUtil.getInstance().getBaseHost());
            entity.setAttachIds(attrs);
        }
        boolean flag = super.saveOrUpdate(entity, false);
        logger.info("保存成功，订单保存结束<<<<<<<<<<<<<<<<<<<<<<<<<<<<<");
        return flag;
    }

    /**
     * 查询待接收列表
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public IPage<OrderVO> queryOrderWaitList(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName"); // 合同名称
        fuzzyFields.add("projectName"); // 项目名称
        fuzzyFields.add("orgName"); // 采购方名称
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        // 如果查询参数中没有接收状态，默认查询待接收
        if (!param.getParams().containsKey("receiveState")) {
            param.getParams().put("receiveState",
                    new Parameter(QueryParam.EQ, OrderReceiveStateEnum.WAIT_RECEIVE.getCode()));
        }
        // 默认查询未关闭
        if (!param.getParams().containsKey("orderFlag")) {
            param.getParams().put("orderFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));
        }
        // 默认创建时间倒序
        if (param.getOrderMap().isEmpty()) {
            param.getOrderMap().put("createTime", QueryParam.DESC);
        }
        return this.queryOrderList(param);
    }

    /**
     * 查询代送货列表
     *
     * @param param 查询参数
     * @return 查询结果
     */
    @Override
    public IPage<OrderVO> queryOrderDeliveredList(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName"); // 合同名称
        fuzzyFields.add("projectName"); // 项目名称
        fuzzyFields.add("orgName"); // 采购方名称
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));
        Map<String, Parameter> paramMap = param.getParams();
        // 如果查询参数中没有接收状态，默认查询部分接收和全部接收
        if (!paramMap.containsKey("receiveState")) {
            paramMap.put("receiveState",
                    new Parameter(QueryParam.IN, Arrays.asList(OrderReceiveStateEnum.PART_RECEIVE.getCode(),
                            OrderReceiveStateEnum.FULL_RECEIVE.getCode())));
        }
        // 如果查询参数中没有送货状态，默认查询待送货、部分送货
        if (!paramMap.containsKey("deliverState")) {
            paramMap.put("deliverState",
                    new Parameter(QueryParam.IN, Arrays.asList(OrderDeliverStateEnum.WAIT_DELIVERED.getCode(),
                            OrderDeliverStateEnum.PART_DELIVERED.getCode())));
        }
        // 默认查询未关闭
        if (!param.getParams().containsKey("orderFlag")) {
            param.getParams().put("orderFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));
        }
        // 默认创建时间倒序
        if (param.getOrderMap().isEmpty()) {
            param.getOrderMap().put("createTime", QueryParam.DESC);
        }
        return this.queryOrderList(param);
    }

    /**
     * 查询订单列表
     *
     * @param param 查询参数
     * @return 查询结果
     */
    private IPage<OrderVO> queryOrderList(QueryParam param) {
        IPage<OrderEntity> page = super.queryPage(param, false);
        IPage<OrderVO> pageData = new Page<>(page.getCurrent(), page.getSize(), page.getTotal());
        pageData.setRecords(BeanMapper.mapList(page.getRecords(), OrderVO.class));
        return pageData;
    }


    /**
     * 接收、拒收订单
     *
     * @param saveOrUpdateVO 订单详情
     * @return 保存结果
     */
    @Override
    public OrderVO saveOrderDelivered(OrderVO saveOrUpdateVO) {
        if (saveOrUpdateVO.getId() == null) {
            throw new BusinessException("订单信息不存在");
        }
        if (saveOrUpdateVO.getReceiveState() == null
                || OrderReceiveStateEnum.getEnumByStateCode(saveOrUpdateVO.getReceiveState()) == null) {
            throw new BusinessException("接收类型不正确");
        }
//        OrderEntity entity = BeanMapper.map(saveOrUpdateVO, OrderEntity.class);
        // 重新查询订单数据，页面上只需要传入少量数据即可
        OrderEntity entity = super.selectById(saveOrUpdateVO.getId());
        entity.setReceiveReason(saveOrUpdateVO.getReceiveReason()); // 不予接收原因
//        entity.setMemo(saveOrUpdateVO.getMemo()); // 原因
        entity.setReceiveState(saveOrUpdateVO.getReceiveState());
        // 如果不是不予接收，则清单列表不能为空
        if (!OrderReceiveStateEnum.REFUSE_RECEIVE.getCode().equals(saveOrUpdateVO.getReceiveState())
                && CollectionUtils.isEmpty(saveOrUpdateVO.getOrderDetailList())) {
            throw new BusinessException("清单列表不能为空！");
        }

        // 页面上传入的接收数量
        Map<Long, BigDecimal> voReceiveMap = new HashMap<>();
        for (OrderDetailVO detailVO : saveOrUpdateVO.getOrderDetailList()) {
            voReceiveMap.put(detailVO.getId(), detailVO.getReceiveNumsSum());
        }

        // 对材料清单进行处理，赋默认值
        if (CollectionUtils.isNotEmpty(entity.getOrderDetailList())) {
            for (OrderDetailEntity detail : entity.getOrderDetailList()) {
                // 全部接收，接收数量等于订单数量
                if (OrderReceiveStateEnum.FULL_RECEIVE.getCode().equals(entity.getReceiveState())) {
                    detail.setReceiveNumsSum(detail.getOrderNumSum());
                }
                // 部分接收，接收数量取传入的接收数量
                else if (OrderReceiveStateEnum.PART_RECEIVE.getCode().equals(entity.getReceiveState())) {
                    detail.setReceiveNumsSum(CommonUtils.setBigDecimalDefaultValue(voReceiveMap.get(detail.getId())));
                }
                // 不予接收默认值0
                else if (OrderReceiveStateEnum.REFUSE_RECEIVE.getCode().equals(entity.getReceiveState())) {
                    detail.setReceiveNumsSum(BigDecimal.ZERO);
                }
                // 未发货数量等于接收数量
                detail.setNotShippedNumSum(detail.getReceiveNumsSum());
                // 发货数量等于0
                detail.setDeliverNumsSum(BigDecimal.ZERO);
            }
            entity.setDeliverState(OrderDeliverStateEnum.WAIT_DELIVERED.getCode());
        }

        // 通知消耗材订单接收状态
        logger.info("开始通知施工方订单信息>>>>>>>>>>>>>>>>>>>");
//        OrderVO vo = BeanMapper.map(entity, OrderVO.class);
        OrderVO vo = new OrderVO();
        vo.setId(entity.getId());
        vo.setReceiveState(entity.getReceiveState());
        List<OrderDetailVO> detailList = new ArrayList<>();
        for (OrderDetailEntity detail : entity.getOrderDetailList()) {
            OrderDetailVO detailVO = new OrderDetailVO();
            detailVO.setId(detail.getId());
            detailVO.setOrderId(detail.getOrderId());
            detailVO.setReceiveNumsSum(detail.getReceiveNumsSum());
            detailList.add(detailVO);
        }
        vo.setOrderDetailList(detailList);
//        vo.setMaterialOrderDetailList(entity.getMaterialOrderDetailList());
//        String url = pushMaterialOrder;
        String url = InteractiveConstants.PUSH_MATERIAL_ORDER;
        String postData = JSONObject.toJSONString(vo);
        String systemId = entity.getSystemId();
        logger.info("发送参数===url:{},postData:[{}],systemId:{}", url, postData, systemId);
        CommonResponse<String> commonResponse = systemDataPushService
                .exchangeDataWithThirdSystem(url, RequestMethod.POST, postData, systemId);
        CommonUtils.checkCommonResponse(commonResponse, logger);
        logger.info("通知施工方订单信息结束<<<<<<<<<<<<<<<<<<<<<<");
        super.saveOrUpdate(entity, false);
        return vo;
    }

    /**
     * 施工方撤回时删除订单
     *
     * @param vo 需要删除的订单
     * @return 删除结果
     */
    @Override
    public boolean deleteOrder(OrderVO vo) {
        logger.info("进入施工方撤回订单接口>>>>>>>>>>>>>>>>>>>");
        logger.info("接收到数据：{}", JSONObject.toJSONString(vo));
        if (null == vo.getId()) {
            throw new BusinessException("订单信息不存在");
        }
        OrderEntity checkEntity = super.selectById(vo.getId());
        if (null == checkEntity) {
            throw new BusinessException("订单信息不存在");
        }
        if (!OrderReceiveStateEnum.WAIT_RECEIVE.getCode().equals(checkEntity.getReceiveState())) {
            throw new BusinessException("订单已被处理，无法删除！");
        }
        CommonResponse<List<AttachmentVO>> fileResp = attachmentApi
                .queryListBySourceId(checkEntity.getId(), materialOrderBillType, fileSourceType, null);
        // 查询成功并有相应的文件
        if (fileResp.isSuccess() && CollectionUtils.isNotEmpty(fileResp.getData())) {
            logger.info("删除文件信息：{}", JSONObject.toJSONString(fileResp.getData()));
            String ids = fileResp.getData().stream().map(AttachmentVO::getId).map(String::valueOf)
                    .collect(Collectors.joining(","));
            attachmentApi.delete(ids);
        }
        boolean flag = baseMapper.deleteOrder(vo.getId()) && orderDetailService.deleteOrderDetailByOrderId(vo.getId());
        logger.info("施工方撤回订单成功<<<<<<<<<<<<<<<<<<<<<<<");
        return flag;
    }


    /**
     * 施工方关闭订单
     *
     * @param vo 需要关闭订单
     * @return 关闭订单
     */
    @Override
    public boolean updateOrderCloseFlag(OrderVO vo) {
        logger.info("进入施工方关闭订单接口>>>>>>>>>>>>>>>>>>>");
        logger.info("接收到数据：{}", JSONObject.toJSONString(vo));
        if (null == vo.getId()) {
            throw new BusinessException("订单信息不存在");
        }
        OrderEntity orderEntity = super.selectById(vo.getId());
        if (CloseFlagEnum.CLOSE.getCode().equals(orderEntity.getOrderFlag())) {
            throw new BusinessException("订单已被关闭！");
        }
        // 查询是否存在未生效的送货单
        // 查询是否存在审批未通过或审批通过待验收的送货单，
        QueryParam queryParam = new QueryParam();
        queryParam.getParams().put("orderId", new Parameter(QueryParam.EQ, orderEntity.getId()));

        // 修改为只查询审批通过待验收未关闭的送货单
        queryParam.getParams().put("checkStatus", new Parameter(QueryParam.EQ, DeliverCheckStateEnum.WAIT_CHECK.getCode()));
        queryParam.getParams().put("billState", new Parameter(QueryParam.IN,
                Arrays.asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
                        BillStateEnum.PASSED_STATE.getBillStateCode())));
        queryParam.getParams().put("closeFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));

//        ComplexParam complexParam = new ComplexParam();
//        complexParam.setLogic(ComplexParam.AND);
//        ComplexParam c1 = new ComplexParam();
//        c1.setLogic(ComplexParam.OR);
//        // 审批未通过
//        c1.getParams().put("billState", new Parameter(QueryParam.IN, Arrays
//                .asList(BillStateEnum.UNCOMMITED_STATE.getBillStateCode(),
//                        BillStateEnum.APPROVING_HAS_STATE.getBillStateCode(),
//                        BillStateEnum.UNAPPROVED.getBillStateCode(),
//                        BillStateEnum.APPROVING_UNEXAM_STATE.getBillStateCode())));
//        complexParam.getComplexParams().add(c1);

        // 审批通过且待验收未关闭
//        ComplexParam c2 = new ComplexParam();
//        c2.setLogic(ComplexParam.OR);
//        c2.getParams().put("checkStatus", new Parameter(QueryParam.EQ, DeliverCheckStateEnum.WAIT_CHECK.getCode()));
//        ComplexParam c3 = new ComplexParam();
//        c3.setLogic(ComplexParam.AND);
//        c3.getParams().put("billState", new Parameter(QueryParam.IN, Arrays
//                .asList(BillStateEnum.COMMITED_STATE.getBillStateCode(),
//                        BillStateEnum.PASSED_STATE.getBillStateCode())));
//        ComplexParam c4 = new ComplexParam();
//        c4.setLogic(ComplexParam.AND);
//        c4.getParams().put("closeFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));
//        c2.getComplexParams().add(c4);
//
//        complexParam.getComplexParams().add(c2);
//        queryParam.getComplexParams().add(complexParam);
        List<DeliveryEntity> deliveryEntityList = deliveryService.queryList(queryParam);
        if (CollectionUtils.isNotEmpty(deliveryEntityList)) {
            logger.error("存在未生效的送货单，送货单信息：{}", JSONObject.toJSONString(deliveryEntityList));
            StringBuilder errorBillCode = new StringBuilder("发货单[");
            for (DeliveryEntity deliveryEntity : deliveryEntityList) {
                errorBillCode.append(deliveryEntity.getBillCode()).append(",");
            }
            errorBillCode.delete(errorBillCode.length() - 1, errorBillCode.length());
            errorBillCode.append("]未完成验收，不能关闭订单");
            throw new BusinessException(errorBillCode.toString());
        }
        // 校验都通过，则修改为已关闭
        orderEntity.setOrderFlag(CloseFlagEnum.CLOSE.getCode());
        boolean flag = super.saveOrUpdate(orderEntity, false);
        logger.info("施工方关闭订单成功>>>>>>>>>>>>>>>>>>>");
        return flag;
    }

    /**
     * 查询订单明细及订单下的送货单
     *
     * @param id 订单
     * @return 查询订单
     */
    @Override
    public OrderVO queryOrderDetail(Long id) {
        OrderEntity orderEntity = super.selectById(id);
        OrderVO vo = BeanMapper.map(orderEntity, OrderVO.class);
        // 部分接收、全部接收需要处理送货单数据
        if (OrderReceiveStateEnum.FULL_RECEIVE.getCode().equals(vo.getReceiveState())
                || OrderReceiveStateEnum.PART_RECEIVE.getCode().equals(vo.getReceiveState())) {
            Map<Long, BigDecimal> deliveryMap = deliveryService.sumDeliveredGroupByDetailId(id);
            for (OrderDetailVO detailVO : vo.getOrderDetailList()) {
                BigDecimal deliverNumsSum = CommonUtils.setBigDecimalDefaultValue(deliveryMap.get(detailVO.getId()));
                detailVO.setDeliverNumsSum(deliverNumsSum);// 已发货数量
                detailVO.setNotShippedNumSum(detailVO.getReceiveNumsSum().subtract(deliverNumsSum));
            }
            QueryParam queryParam = new QueryParam();
            queryParam.getParams().put("orderId", new Parameter(QueryParam.EQ, id));
            queryParam.getOrderMap().put("createTime", QueryParam.DESC);
            List<DeliveryEntity> deliveryEntityList = deliveryService.queryList(queryParam);
            if (CollectionUtils.isNotEmpty(deliveryEntityList)) {
                List<DeliveryVO> deliveryVOList = BeanMapper
                        .mapList(deliveryEntityList, DeliveryVO.class);
                if (CollectionUtils.isNotEmpty(deliveryVOList)) {
                    for (DeliveryVO deliveryVO : deliveryVOList) {
                        deliveryVO.setBillStateName(BillStateEnum.getEnumByStateCode(deliveryVO.getBillState())
                                .getDescription());
                    }
                }
                vo.setDeliveryList(deliveryVOList);
            }
        }
        return vo;
    }

    @Override
    public IPage<OrderVO> queryAllList(QueryParam param) {
        /** 模糊搜索配置字段示例 */
        List<String> fuzzyFields = param.getFuzzyFields();
        fuzzyFields.add("contractName"); // 合同名称
        fuzzyFields.add("projectName"); // 项目名称
        fuzzyFields.add("orgName"); // 采购方名称
        /** 租户隔离 */
        param.getParams().put("tenantId", new Parameter(QueryParam.EQ, InvocationInfoProxy.getTenantid()));

        param.getParams().put("receiveState",
                new Parameter(QueryParam.IN, Arrays.asList(OrderReceiveStateEnum.WAIT_RECEIVE.getCode(), OrderReceiveStateEnum.PART_RECEIVE.getCode(),
                        OrderReceiveStateEnum.FULL_RECEIVE.getCode())));
        // 默认查询未关闭
        if (!param.getParams().containsKey("orderFlag")) {
            param.getParams().put("orderFlag", new Parameter(QueryParam.EQ, CloseFlagEnum.NORMAL.getCode()));
        }
        // 默认创建时间倒序
        if (param.getOrderMap().isEmpty()) {
            param.getOrderMap().put("createTime", QueryParam.DESC);
        }
        return this.queryOrderList(param);
    }

}
